##############################################
# Plantilla análisis básico en R
# Autor: Nabil Subhi-Issa, PhD
# 1) Importar dataset (Excel o CSV)
# 2) Comprobar normalidad y homocedasticidad
# 3) t de Student / Mann-Whitney para 2 grupos (variables numéricas)
# 4) ANOVA / Kruskal-Wallis para ≥3 grupos (variables numérica)
# + Soporte para muchas variables y variables categóricas
##############################################

#### 0. Paquetes necesarios ----
# install.packages(c("readr", "readxl", "dplyr", "car"))

library(readr)
library(readxl)
library(dplyr)
library(car)
library(tibble)
library(writexl)

#### 1. CONFIGURACIÓN DEL USUARIO ----
# >>> Cambia estas variables según tu análisis <<<

file_path <- "Database_R.xlsx" # Ruta al archivo .csv o .xlsx
file_type  <- "excel"                 # "csv" o "excel"

setwd("") #especifica la ruta a una carpeta
getwd()    #Para saber si R busca en el sitio correcto

# Variable de grupo (categórica, ej. tratamiento, grupo)
group_var    <- "Dx"
strat_var  <- "Phase"  # Variable de estratificación adicional. Debe ser factor/categórica


alpha <- 0.05                       # Nivel de significación

# Opciones de exportación
export_folder         <- "resultados_analisis" # puedes poner el nombre que quieras a la carpeta
export_descriptives   <- TRUE   # Exportar resumen descriptivo
export_results_table  <- TRUE   # Exportar tabla de resultados
export_excel_also     <- TRUE   # Además de CSV, exportar a Excel

#### 2. IMPORTAR DATOS ----

if (file_type == "csv") {
  database <- read_csv(file_path, show_col_types = FALSE)
} else if (file_type == "excel") {
  database <- read_excel(file_path)
} else {
  stop("file_type debe ser 'csv' o 'excel'")
}

# Convertir character (letras) en factor
database <- database %>%
  mutate(across(where(is.character), as.factor))
# Convertir automáticamente variables 0/1 en factores (sí/no)
binary_manual <- ""  # adapta a tu caso

database <- database %>%
  mutate(across(
    all_of(binary_manual),
    ~ factor(.x, levels = c(0, 1), labels = c("no", "si"))
  ))

# Detectar tipos de variables
numeric_vars <- names(Filter(is.numeric, database))
factor_vars  <- names(Filter(is.factor, database))

cat("Variables NUMÉRICAS detectadas:\n")
print(numeric_vars)
cat("\nVariables CATEGÓRICAS (factor) detectadas:\n")
print(factor_vars)
cat("\n----------------------------------------\n\n")

# Evitar usar como respuesta directa variables que son de agrupación
numeric_response_candidates <- setdiff(numeric_vars, group_var)

if (length(numeric_response_candidates) == 0) {
  stop("No hay variables numéricas disponibles para analizar (tras excluir group_var).")
}

# Crear dataset limpio (elimiar NAs) para el análisis principal, en caso de que
# no quieras NAs en tu dataset. Recomiendo saltarlo para estadística básica
data_clean <- database %>%
  select(all_of(c(response_var, group_var))) %>%
  na.omit()

#### Resumen rápido de TODAS las variables numéricas por grupo ----

cat("Resumen descriptivo de TODAS las variables numéricas por grupo:\n")

resumen_numericas <- database %>%
  group_by(.data[[group_var]]) %>%
  summarise(
    n = n(),
    across(
      .cols = where(is.numeric),
      .fns  = list(
        mean   = ~mean(.x, na.rm = TRUE),
        sd     = ~sd(.x, na.rm = TRUE),
        median = ~median(.x, na.rm = TRUE),
        IQR    = ~IQR(.x, na.rm = TRUE)
      ),
      .names = "{.col}_{.fn}"
    ),
    .groups = "drop"
  )

# Exportar resumen descriptivo
if (export_descriptives) {
  csv_path  <- file.path(export_folder, "resumen_numericas_por_grupo.csv")
  write_csv(resumen_numericas, csv_path)
  
  if (export_excel_also) {
    xlsx_path <- file.path(export_folder, "resumen_numericas_por_grupo.xlsx")
    write_xlsx(list(resumen_numericas = resumen_numericas), path = xlsx_path)
  }
}
# Si no funciona la exportación, asegúrate que la carpeta existe, en la ruta que has
# especificado anteriormente en "file_path"
dir.exists("resultados_analisis")


#### 4. FUNCIÓN PARA COMPROBAR SUPUESTOS (para UNA variable) ----

check_assumptions <- function(df, response, group, alpha = 0.05) {
  g <- df[[group]]
  y <- df[[response]]
  
  n_groups <- length(unique(g))
  
  cat("Número de grupos:", n_groups, "\n\n")
  
  # 4.1 Normalidad (Shapiro-Wilk por grupo, si hay ≥3 datos)
  cat(">>> Test de normalidad (Shapiro-Wilk) por grupo para", response, ":\n")
  normality_results <- df %>%
    group_by(.data[[group]]) %>%
    summarise(
      n       = length(.data[[response]]),
      p_value = ifelse(
        n >= 3,
        shapiro.test(.data[[response]])$p.value,
        NA_real_
      ),
      .groups = "drop"
    )
  print(normality_results)
  cat("\n(Orientación: p > ", alpha, " → no se rechaza normalidad)\n\n", sep = "")
  
  valid_p   <- normality_results$p_value[!is.na(normality_results$p_value)]
  all_normal <- length(valid_p) > 0 && all(valid_p > alpha)
  
  # 4.2 Homocedasticidad (Levene)
  cat(">>> Test de homogeneidad de varianzas (Levene) para", response, ":\n")
  levene <- leveneTest(df[[response]] ~ df[[group]])
  print(levene)
  cat("\n(Interpretación: p > ", alpha, " → no se rechaza homocedasticidad)\n\n", sep = "")
  
  homo <- levene$`Pr(>F)`[1] > alpha
  
  list(
    n_groups    = n_groups,
    normality   = normality_results,
    all_normal  = all_normal,
    homogeneity = homo
  )
}

#### 5. TABLA GLOBAL DE RESULTADOS ----

results_table <- tibble(
  analysis_type = character(),  # "numeric" o "categorical"
  variable      = character(),  # nombre de la variable analizada
  group_var     = character(),  # nombre de la variable de grupo
  comparacion   = character(),  # NA / "global" / "pairwise"
  group1        = character(),  # solo para categóricas pairwise
  group2        = character(),  # solo para categóricas pairwise
  n_groups      = integer(),    # nº niveles de group_var en ese análisis
  test_used     = character(),  # nombre del test
  p_value       = numeric(),    # p-valor
  note          = character()   # comentario (paramétrico, no paramétrico, etc.)
)

#### 6. FUNCIÓN PARA LANZAR EL TEST ADECUADO (UNA variable numérica) ----

run_comparison <- function(df,
                           response,
                           group,
                           alpha = 0.05,
                           strat_var_name = NA_character_,
                           strat_level = NA_character_) {
  
  df_local <- df %>%
    select(all_of(c(response, group))) %>%
    na.omit()
  
  if (nrow(df_local) == 0) {
    return(tibble(
      analysis_type = "numeric",
      variable      = response,
      group_var     = group,
      comparacion   = NA_character_,
      group1        = NA_character_,
      group2        = NA_character_,
      n_groups      = NA_integer_,
      test_used     = NA_character_,
      p_value       = NA_real_,
      note          = "sin datos"
    ))
  }
  
  n_groups <- length(unique(df_local[[group]]))

  if (n_groups < 2) {
    return(tibble(
      analysis_type = "numeric",
      variable      = response,
      group_var     = group,
      comparacion   = NA_character_,
      group1        = NA_character_,
      group2        = NA_character_,
      n_groups      = n_groups,
      test_used     = NA_character_,
      p_value       = NA_real_,
      note          = "solo 1 grupo"
    ))
  }
  
  sup <- check_assumptions(df_local, response, group, alpha)
  
  test_name <- NA_character_
  p_val     <- NA_real_
  note      <- ""
  
  if (n_groups == 2) {
    
    if (sup$all_normal && sup$homogeneity) {
      result <- t.test(df_local[[response]] ~ df_local[[group]], var.equal = TRUE)
      test_name <- "t_student"
      p_val     <- result$p.value
      note      <- "paramétrico"
    } else {
      result <- wilcox.test(df_local[[response]] ~ df_local[[group]], exact = FALSE)
      test_name <- "mann_whitney"
      p_val     <- result$p.value
      note      <- "no paramétrico"
    }
    
  } else {
    
    if (sup$all_normal && sup$homogeneity) {
      modelo <- aov(df_local[[response]] ~ df_local[[group]])
      sum_aov <- summary(modelo)
      p_val   <- sum_aov[[1]][["Pr(>F)"]][1]
      test_name <- "ANOVA"
      note      <- "paramétrico"
    } else {
      result <- kruskal.test(df_local[[response]] ~ df_local[[group]])
      p_val     <- result$p.value
      test_name <- "kruskal_wallis"
      note      <- "no paramétrico"
    }
  }
  
  tibble(
    analysis_type = "numeric",
    variable      = response,
    group_var     = group,
    comparacion   = NA_character_,
    group1        = NA_character_,
    group2        = NA_character_,
    n_groups      = n_groups,
    test_used     = test_name,
    p_value       = p_val,
    note          = note
  )
}

#### 7. BUCLE PRINCIPAL: ANÁLISIS PARA TODAS LAS VARIABLES NUMÉRICAS

  # Loop sobre cada variable numérica candidata
  for (var in numeric_response_candidates) {
    res_row <- run_comparison(
      df              = database,
      response        = var,
      group           = group_var,
      alpha           = alpha
    )
    
    results_table <- bind_rows(results_table, res_row)
  }

#### 7. EXPORTAR TABLA GLOBAL DE RESULTADOS ----

if (export_results_table && nrow(results_table) > 0) {
  csv_path  <- file.path(export_folder, "resultados_tests.csv")
  write_csv(results_table, csv_path)
  
  if (export_excel_also) {
    xlsx_path <- file.path(export_folder, "resultados_tests.xlsx")
    write_xlsx(list(resultados = results_table), path = xlsx_path)
  }
}

#### 8. EXTRA (OPCIONAL): ANÁLISIS ENTRE VARIABLES CATEGÓRICAS ----
# Útil para cosas tipo:
#   fumador (sí/no) vs evento_clínico (sí/no)
#   grupo_tratamiento vs respuesta_clínica (respondedor/no respondedor)

###############################################################
#### 9. ANÁLISIS ENTRE VARIABLES CATEGÓRICAS (group_var vs otra)
# - Test global (todos los grupos)
# - Pairwise (dos a dos) si hay ≥ 3 grupos
###############################################################

analyze_categorical_pairs <- function(df, group_var, cat_var, alpha = 0.05) {
  
  # Comprobaciones
  if (!group_var %in% names(df)) stop("group_var no existe en el dataset.")
  if (!cat_var   %in% names(df)) stop("cat_var no existe en el dataset.")
  
  # Asegurar factores
  df[[group_var]] <- as.factor(df[[group_var]])
  df[[cat_var]]   <- as.factor(df[[cat_var]])
  
  # Quitar filas con NA en esas dos variables
  df_local <- df %>%
    select(all_of(c(group_var, cat_var))) %>%
    na.omit()
  
  niveles_grupo <- levels(df_local[[group_var]])
  n_groups <- length(niveles_grupo)
  
  if (n_groups < 2) stop("Se necesita al menos 2 niveles en group_var.")
  
  # ---- 1) TEST GLOBAL ----
  tab_global <- table(df_local[[group_var]], df_local[[cat_var]])
  
  cat("\n=========================================\n")
  cat("Análisis categórico global:", group_var, "vs", cat_var, "\n")
  cat("=========================================\n")
  cat("Tabla de contingencia (global):\n")
  print(tab_global)
  
  chisq_obj <- suppressWarnings(chisq.test(tab_global))
  chisq_ok  <- all(chisq_obj$expected >= 5)
  
  if (chisq_ok) {
    cat("\nUsando Chi-cuadrado global (frecuencias esperadas ≥ 5)\n")
    print(chisq_obj)
    p_global   <- chisq_obj$p.value
    test_global <- "chi_cuadrado_global"
    note_global <- "frecuencias esperadas ≥ 5"
  } else {
    cat("\nUsando test exacto de Fisher global (frecuencias bajas)\n")
    fisher_obj <- fisher.test(tab_global)
    print(fisher_obj)
    p_global   <- fisher_obj$p.value
    test_global <- "fisher_global"
    note_global <- "frecuencias esperadas bajas"
  }
  
  # Guardamos resultados en un tibble
  results_cat <- tibble(
    comparacion = "global",
    group1      = NA_character_,
    group2      = NA_character_,
    variable_grupo = group_var,
    variable_cat   = cat_var,
    test_used      = test_global,
    p_value        = p_global,
    note           = note_global
  )
  
  # ---- 2) PAIRWISE DOS A DOS (si hay ≥ 3 grupos) ----
  if (n_groups >= 3) {
    cat("\n\n=========================================\n")
    cat("Análisis pairwise (dos a dos) por niveles de", group_var, "\n")
    cat("=========================================\n")
    
    pares <- t(combn(niveles_grupo, 2))  # todas las combinaciones de 2 grupos
    
    for (i in seq_len(nrow(pares))) {
      g1 <- pares[i, 1]
      g2 <- pares[i, 2]
      
      cat("\n-----------------------------------------\n")
      cat("Comparación:", g1, "vs", g2, " | Variable:", cat_var, "\n")
      cat("-----------------------------------------\n")
      
      df_pair <- df_local %>%
        filter(.data[[group_var]] %in% c(g1, g2))
      
      # Re-dropear niveles no usados (por estética)
      df_pair[[group_var]] <- droplevels(df_pair[[group_var]])
      
      tab_pair <- table(df_pair[[group_var]], df_pair[[cat_var]])
      cat("Tabla de contingencia:\n")
      print(tab_pair)
      
      chisq_obj_pair <- suppressWarnings(chisq.test(tab_pair))
      chisq_ok_pair  <- all(chisq_obj_pair$expected >= 5)
      
      if (chisq_ok_pair) {
        cat("\nUsando Chi-cuadrado (frecuencias esperadas ≥ 5)\n")
        print(chisq_obj_pair)
        p_val   <- chisq_obj_pair$p.value
        test_pw <- "chi_cuadrado_pairwise"
        note_pw <- "frecuencias esperadas ≥ 5"
      } else {
        cat("\nUsando test exacto de Fisher (frecuencias bajas)\n")
        fisher_obj_pair <- fisher.test(tab_pair)
        print(fisher_obj_pair)
        p_val   <- fisher_obj_pair$p.value
        test_pw <- "fisher_pairwise"
        note_pw <- "frecuencias esperadas bajas"
      }
      
      results_cat <- bind_rows(
        results_cat,
        tibble(
          comparacion    = "pairwise",
          group1         = g1,
          group2         = g2,
          variable_grupo = group_var,
          variable_cat   = cat_var,
          test_used      = test_pw,
          p_value        = p_val,
          note           = note_pw
        )
      )
    }
  }
  
  results_cat
}

# Para lanzar el análisis
res_categorico <- analyze_categorical_pairs(data, group_var = "", cat_var = "")
res_categorico
